//	Zinc Interface Library - WINDSP.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include "ui_dsp.hpp"
#pragma hdrstop

HDC UI_MSWINDOWS_DISPLAY::hDC = 0;
HFONT UI_MSWINDOWS_DISPLAY::fontTable[MAX_LOGICAL_FONTS] =
{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

// ----- UI_MSWINDOWS_DISPLAY -----------------------------------------------

UI_MSWINDOWS_DISPLAY::UI_MSWINDOWS_DISPLAY(HANDLE _hInstance,
	HANDLE _hPrevInstance, int _nCmdShow) :
	UI_DISPLAY(FALSE), maxColors(16)
{
	extern void z_windsp_dummy(void);		// Bug fix for Zortech & Microsoft linkers.
	z_windsp_dummy();

	hInstance = _hInstance;
	hPrevInstance = _hPrevInstance;
	nCmdShow = _nCmdShow;

	// Set up the initial logical fonts.
	LONG baseUnits = GetDialogBaseUnits();
	HFONT systemFont = GetStockObject(SYSTEM_FONT);
	LOGFONT info;
	GetObject(systemFont, sizeof(LOGFONT), (LPSTR)&info);
	HFONT smallFont = CreateFont(HIWORD(baseUnits) * 3 / 4,
		LOWORD(baseUnits) * 3 / 4, info.lfEscapement, info.lfOrientation,
		400, info.lfItalic, info.lfUnderline, info.lfStrikeOut,
		info.lfCharSet, info.lfOutPrecision, info.lfClipPrecision,
		info.lfQuality, info.lfPitchAndFamily, "Helv");
	if (!fontTable[FNT_SMALL_FONT])
		fontTable[FNT_SMALL_FONT] = smallFont;
	HFONT dialogFont = CreateFont(HIWORD(baseUnits) * 3 / 4,
		LOWORD(baseUnits) * 3 / 4, info.lfEscapement, info.lfOrientation,
		info.lfWeight, info.lfItalic, info.lfUnderline, info.lfStrikeOut,
		info.lfCharSet, info.lfOutPrecision, info.lfClipPrecision,
		info.lfQuality, info.lfPitchAndFamily, "Helv");
	if (!fontTable[FNT_DIALOG_FONT])
		fontTable[FNT_DIALOG_FONT] = dialogFont;
	for (int i = FNT_SYSTEM_FONT; i < MAX_LOGICAL_FONTS; i++)
		if (!fontTable[i])
			fontTable[i] = systemFont;

	// Compute the default display values.
	GetObject(dialogFont, sizeof(LOGFONT), (LPSTR)&info);
	columns = GetSystemMetrics(SM_CXSCREEN);
	lines = GetSystemMetrics(SM_CYSCREEN);
	cellWidth = info.lfWidth + 1;
//	cellHeight = GetSystemMetrics(SM_CYMENU) + preSpace + postSpace + 2;
	cellHeight =
		info.lfHeight - 1 +		// average letter height minus a funny fudge factor
		info.lfHeight / 3 +		// blank space above the letter
		info.lfHeight / 4 +		// blank space below the letter
		2 +						// border space
		preSpace +				// preSpace (before the border)
		postSpace;				// postSpace (after the border)
	installed = TRUE;
}

UI_MSWINDOWS_DISPLAY::~UI_MSWINDOWS_DISPLAY(void)
{
	DeleteObject(fontTable[FNT_SMALL_FONT]);
	DeleteObject(fontTable[FNT_DIALOG_FONT]);
}

void UI_MSWINDOWS_DISPLAY::Bitmap(SCREENID screenID, int column, int line,
	int bitmapWidth, int bitmapHeight, const UCHAR *bitmapArray,
	const UI_PALETTE *palette, const UI_REGION *clipRegion)
{
	// Get the device context.
	HDC _hDC = (screenID != ID_DIRECT) ? GetDC(screenID) : hDC;
	int oldROP2 = SetROP2(_hDC, R2_COPYPEN);

	// Set the clipRegion (Relative to _hDC).
	HRGN cRegion = 0;
	if (clipRegion)
	{
		cRegion = CreateRectRgn(clipRegion->left, clipRegion->top,
			clipRegion->right + 1, clipRegion->bottom + 1);
		SelectClipRgn(_hDC, cRegion);
	}

	UCHAR *pixel = (UCHAR *)bitmapArray;
	int left = column;
	int top = line;
	int right = column + bitmapWidth;
	int bottom = line + bitmapHeight;

	for (int y = top; y < bottom; y++)
	{
		for (int x = left; x < right; x++, pixel++)
			if (*pixel != BACKGROUND)
				if (palette)
					SetPixel(_hDC, x, y, MapColor(&palette[*pixel], TRUE));
				else
					SetPixel(_hDC, x, y, MapColor(&colorMap[*pixel], TRUE));
	}

	SetROP2(_hDC, oldROP2);
	if (cRegion)
		DeleteObject(cRegion);
	if (screenID != ID_DIRECT)
		ReleaseDC(screenID, _hDC);
}

void UI_MSWINDOWS_DISPLAY::Ellipse(SCREENID screenID, int x, int y,
	int startAngle, int endAngle, int xRadius, int yRadius,
	const UI_PALETTE *palette, int fill, int xor, const UI_REGION *clipRegion)
{
	const int pSin[] = { 0, 871, 1736, 2588, 3420, 4226, 5000, 5736, 6428,
		7071, 7660, 8191, 8660, 9063, 9397, 9659, 9848, 9962, 10000 };

	// Get the device context.
	HDC _hDC = (screenID != ID_DIRECT) ? GetDC(screenID) : hDC;
	int oldROP2 = SetROP2(_hDC, (xor == TRUE) ? R2_XORPEN : R2_COPYPEN);

	// Set the clipRegion (Relative to _hDC).
	HRGN cRegion = 0;
	if (clipRegion)
	{
		cRegion = CreateRectRgn(clipRegion->left, clipRegion->top,
			clipRegion->right + 1, clipRegion->bottom + 1);
		SelectClipRgn(_hDC, cRegion);
	}

	// Compute points to start and stop.
	int startX = 0;
	int startY = 0;
	int endX = 0;
	int endY = 0;
	if (startAngle <= 90)
	{
		startX = x + (int)(1L * pSin[(90 - startAngle) / 5] * xRadius / 10000L);
		startY = y + (int)(1L * pSin[startAngle / 5] * yRadius / 10000L);
	}
	else if (startAngle <= 180)
	{
		startX = x - (int)(1L * pSin[(startAngle - 90) / 5] * xRadius / 10000L);
		startY = y + (int)(1L * pSin[(180 - startAngle) / 5] * yRadius / 10000L);
	}
	else if (startAngle <= 270)
	{
		startX = x - (int)(1L * pSin[(270 - startAngle) / 5] * xRadius / 10000L);
		startY = y - (int)(1L * pSin[(startAngle - 180) / 5] * yRadius / 10000L);
	}
	else if (startAngle <= 360)
	{
		startX = x + (int)(1L * pSin[(startAngle - 270) / 5] * xRadius / 10000L);
		startY = y - (int)(1L * pSin[(360 - startAngle) / 5] * yRadius / 10000L);
	}

	if (endAngle <= 90)
	{
		endX = x + (int)(1L * pSin[(90 - endAngle) / 5] * xRadius / 10000L);
		endY = y + (int)(1L * pSin[endAngle / 5] * yRadius / 10000L);
	}
	else if (endAngle <= 180)
	{
		endX = x - (int)(1L * pSin[(endAngle - 90) / 5] * xRadius / 10000L);
		endY = y + (int)(1L * pSin[(180 - endAngle) / 5] * yRadius / 10000L);
	}
	else if (endAngle <= 270)
	{
		endX = x - (int)(1L * pSin[(270 - endAngle) / 5] * xRadius / 10000L);
		endY = y - (int)(1L * pSin[(endAngle - 180) / 5] * yRadius / 10000L);
	}
	else if (endAngle <= 360)
	{
		endX = x + (int)(1L * pSin[(endAngle - 270) / 5] * xRadius / 10000L);
		endY = y - (int)(1L * pSin[(360 - endAngle) / 5] * yRadius / 10000L);
	}

	HANDLE oldPen = SelectObject(_hDC, CreatePen(PS_SOLID, 1, MapColor(palette, TRUE)));

	if (fill)
	{
		HBITMAP hBitmap = CreateBitmap(8, 8, 1, 1, (LPSTR)patternTable[palette->fillPattern]);
		HBRUSH oldBrush = SelectObject(_hDC, CreatePatternBrush(hBitmap));
		COLORREF oldForeground = SetTextColor(_hDC, MapColor(palette, TRUE));
		COLORREF oldBackground = SetBkColor(_hDC, MapColor(palette, FALSE));
		int oldMode = SetBkMode(_hDC, OPAQUE);

		if (startAngle == 0 && endAngle == 360)
			::Ellipse(_hDC, x - xRadius, y - yRadius, x + xRadius, y + yRadius);
		else
			Pie(_hDC, x - xRadius, y - yRadius, x + xRadius, y + yRadius, endX, endY, startX, startY);

		SetBkMode(_hDC, oldMode);
		SetBkColor(_hDC, oldBackground);
		SetTextColor(_hDC, oldForeground);
		DeleteObject(SelectObject(_hDC, oldBrush));
		DeleteObject(hBitmap);
	}
	else
		// Draw a line ellipse.
		Arc(_hDC, x - xRadius, y - yRadius, x + xRadius, y + yRadius, endX, endY, startX, startY);

	SetROP2(_hDC, oldROP2);
	DeleteObject(SelectObject(_hDC, oldPen));
	if (cRegion)
		DeleteObject(cRegion);
	if (screenID != ID_DIRECT)
		ReleaseDC(screenID, _hDC);
}

void UI_MSWINDOWS_DISPLAY::DeviceMove(IMAGE_TYPE, int, int)
{
}

void UI_MSWINDOWS_DISPLAY::DeviceSet(IMAGE_TYPE, int, int, int, int, UCHAR *)
{
}

void UI_MSWINDOWS_DISPLAY::Line(SCREENID screenID, int x1, int y1,
	int x2, int y2, const UI_PALETTE *palette, int width, int xor,
	const UI_REGION *clipRegion)
{
	// Get the device context.
	HDC _hDC = (screenID != ID_DIRECT) ? GetDC(screenID) : hDC;
	int oldROP2 = SetROP2(_hDC, xor ? R2_XORPEN : R2_COPYPEN);

	// Set the clipRegion (Relative to _hDC).
	HRGN cRegion = 0;
	if (clipRegion)
	{
		cRegion = CreateRectRgn(clipRegion->left, clipRegion->top,
			clipRegion->right + 1, clipRegion->bottom + 1);
		SelectClipRgn(_hDC, cRegion);
	}

	// Draw the line on the window.
	COLORREF color = MapColor(palette, TRUE);
	HPEN foreground = CreatePen(PS_SOLID, width, color);
	HPEN oldForeground = SelectObject(_hDC, foreground);
	MoveTo(_hDC, x1, y1);
	LineTo(_hDC, x2, y2);
	SetPixel(_hDC, x2, y2, color);

	DeleteObject(SelectObject(_hDC, oldForeground));

	SetROP2(_hDC, oldROP2);
	if (cRegion)
		DeleteObject(cRegion);
	if (screenID != ID_DIRECT)
		ReleaseDC(screenID, _hDC);
}

COLOR UI_MSWINDOWS_DISPLAY::MapColor(const UI_PALETTE *palette, int foreground)
{
	// Match the color request based on the COLOR type in fillPattern.
	COLOR color = foreground ? palette->colorForeground : palette->colorBackground;
	if (palette->fillPattern == PTN_SYSTEM_COLOR)
		return (GetSysColor((int)color));
	else if (palette->fillPattern == PTN_RGB_COLOR)
		return (color);
	else if (color <= MAX_COLORMAP_INDEX)
		return (colorMap[(int)color].colorForeground);
	else
		return (RGB_BLACK);
}

void UI_MSWINDOWS_DISPLAY::Polygon(SCREENID screenID, int numPoints,
	const int *polygonPoints, const UI_PALETTE *palette, int fill, int xor,
	const UI_REGION *clipRegion)
{
	// Get the device context.
	HDC _hDC = (screenID != ID_DIRECT) ? GetDC(screenID) : hDC;
	int oldROP2 = SetROP2(_hDC, (xor == TRUE) ? R2_XORPEN : R2_COPYPEN);

	// Set up the points.
	LPPOINT tPolygon = new POINT[numPoints];
	for (int i = 0; i < numPoints; i++)
	{
		tPolygon[i].x = polygonPoints[i*2];
		tPolygon[i].y = polygonPoints[i*2+1];
	}

	// Set the clipRegion.
	HRGN cRegion = 0;
	if (clipRegion)
	{
		cRegion = CreateRectRgn(clipRegion->left, clipRegion->top,
			clipRegion->right + 1, clipRegion->bottom + 1);
		SelectClipRgn(_hDC, cRegion);
	}

	HANDLE oldPen = SelectObject(_hDC, CreatePen(PS_SOLID, 1, MapColor(palette, TRUE)));
	int oldFillMode = SetPolyFillMode(_hDC, WINDING);

	if (fill)
	{
		// Set the fill pattern.
		HBITMAP hBitmap = CreateBitmap(8, 8, 1, 1, (LPSTR)patternTable[palette->fillPattern]);
		HBRUSH oldBrush = SelectObject(_hDC, CreatePatternBrush(hBitmap));
		COLORREF oldForeground = SetTextColor(_hDC, MapColor(palette, TRUE));
		COLORREF oldBackground = SetBkColor(_hDC, MapColor(palette, FALSE));
		int oldMode = SetBkMode(_hDC, OPAQUE);

		// Draw an filled polygon.
		::Polygon(_hDC, tPolygon, numPoints);

		SetBkMode(_hDC, oldMode);
		SetBkColor(_hDC, oldBackground);
		SetTextColor(_hDC, oldForeground);
		DeleteObject(SelectObject(_hDC, oldBrush));
		DeleteObject(hBitmap);
	}
	else
		// Draw a line polygon.
		Polyline(_hDC, tPolygon, numPoints);

	delete tPolygon;

	SetPolyFillMode(_hDC, oldFillMode);
	DeleteObject(SelectObject(_hDC, oldPen));
	SetROP2(_hDC, oldROP2);
	if (cRegion)
		DeleteObject(cRegion);
	if (screenID != ID_DIRECT)
		ReleaseDC(screenID, _hDC);
}

void UI_MSWINDOWS_DISPLAY::Rectangle(SCREENID screenID, int left, int top,
	int right, int bottom, const UI_PALETTE *palette, int width, int fill,
	int xor, const UI_REGION *clipRegion)
{
	// Get the device context.
	HDC _hDC = (screenID != ID_DIRECT) ? GetDC(screenID) : hDC;
	int oldROP2 = SetROP2(_hDC, xor ? R2_XORPEN : R2_COPYPEN);

	// Change coordinates to be relative to _hDC.
	RECT rect = {left, top, right + 1, bottom + 1};

	// Set the clipRegion.
	HRGN cRegion = 0;
	if (clipRegion)
	{
		cRegion = CreateRectRgn(clipRegion->left, clipRegion->top,
			clipRegion->right + 1, clipRegion->bottom + 1);
		SelectClipRgn(_hDC, cRegion);
	}

	// Draw the outline.
	HBRUSH borderBrush = CreateSolidBrush(MapColor(palette, TRUE));
	for (int i = 0; i < width; i++)
	{
		::FrameRect(_hDC, &rect, borderBrush);
		rect.left++;
		rect.top++;
		rect.right--;
		rect.bottom--;
	}
	DeleteObject(borderBrush);

	// Fill the rectangle.
	if (fill)
	{
		// Set the fill pattern.
		HBITMAP hBitmap = CreateBitmap(8, 8, 1, 1, (LPSTR)&patternTable[palette->fillPattern]);
		HBRUSH fillBrush = CreatePatternBrush(hBitmap);
		COLORREF oldForeground = SetTextColor(_hDC, MapColor(palette, TRUE));
		COLORREF oldBackground = SetBkColor(_hDC, MapColor(palette, FALSE));

		FillRect(_hDC, &rect, fillBrush);

		SetBkColor(_hDC, oldBackground);
		SetTextColor(_hDC, oldForeground);
		DeleteObject(fillBrush);
		DeleteObject(hBitmap);
	}

	SetROP2(_hDC, oldROP2);
	if (cRegion)
		DeleteObject(cRegion);
	if (screenID != ID_DIRECT)
		ReleaseDC(screenID, _hDC);
}

// Stub for MS Windows.
void UI_MSWINDOWS_DISPLAY::RectangleXORDiff(const UI_REGION &, const UI_REGION &)
{
}

void UI_MSWINDOWS_DISPLAY::RegionDefine(SCREENID, int, int, int, int)
{
}

void UI_MSWINDOWS_DISPLAY::RegionMove(const UI_REGION &oldRegion, int newColumn,
	int newLine, SCREENID oldScreenID, SCREENID newScreenID)
{
	// Get the device context.
	HDC _hDestDC = (newScreenID != ID_DIRECT) ? GetDC(newScreenID) : hDC;
	HDC _hSrcDC = (oldScreenID != ID_DIRECT) ? GetDC(oldScreenID) : hDC;

	BitBlt(_hDestDC, newColumn, newLine, oldRegion.right - oldRegion.left + 1,
		oldRegion.bottom - oldRegion.top + 1, _hSrcDC, oldRegion.left,
		oldRegion.top, SRCCOPY);

	if (newScreenID != ID_DIRECT)
		ReleaseDC(newScreenID, _hDestDC);
	if (oldScreenID != ID_DIRECT)
		ReleaseDC(oldScreenID, _hSrcDC);
}

void UI_MSWINDOWS_DISPLAY::Text(SCREENID screenID, int left, int top,
	const char *text, const UI_PALETTE *palette, int length, int fill,
	int xor, const UI_REGION *clipRegion, LOGICAL_FONT logicalFont)
{
	logicalFont &= 0x0FFF;
	// Get the device context.
	HDC _hDC = (screenID != ID_DIRECT) ? GetDC(screenID) : hDC;
	int oldDrawMode = SetROP2(_hDC, xor ? R2_XORPEN : R2_COPYPEN);

	// Set the clipRegion.
	HRGN cRegion = 0;
	if (clipRegion)
	{
		cRegion = CreateRectRgn(clipRegion->left, clipRegion->top,
			clipRegion->right + 1, clipRegion->bottom + 1);
		SelectClipRgn(_hDC, cRegion);
	}
	if (length < 0)
		length = ui_strlen(text);

	// Set up the colors, fill and font and draw the text.
	COLORREF oldForeground = SetTextColor(_hDC, MapColor(palette, TRUE));
	int oldMode = SetBkMode(_hDC, TRANSPARENT);
	HFONT oldFont = SelectObject(_hDC, fontTable[logicalFont]);

	if (fill)
		Rectangle(screenID, left, top, left + TextWidth(text, screenID, logicalFont) - 1,
			top + TextHeight(text, screenID, logicalFont) - 1, palette, 0, TRUE,
			xor, clipRegion);

	TextOut(_hDC, left, top, (LPSTR)text, length);

	SetTextColor(_hDC, oldForeground);
	SetBkMode(_hDC, oldMode);
	SelectObject(_hDC, oldFont);
	SetROP2(_hDC, oldDrawMode);
	if (cRegion)
		DeleteObject(cRegion);
	if (screenID != ID_DIRECT)
		ReleaseDC(screenID, _hDC);
}

int UI_MSWINDOWS_DISPLAY::TextHeight(const char *string, SCREENID screenID,
	LOGICAL_FONT logicalFont)
{
	logicalFont &= 0x0FFF;
	HDC _hDC;
	if (screenID != ID_SCREEN)
		_hDC = GetDC(screenID);
	else
	{
		HDC tHDC = GetDC(GetDesktopWindow());
		_hDC = CreateCompatibleDC(tHDC);
		ReleaseDC(GetDesktopWindow(), tHDC);
	}

	// Get the string height.
	HFONT oldFont = SelectObject(_hDC, fontTable[logicalFont]);
	int height = HIWORD(GetTextExtent(_hDC, (LPSTR)string, ui_strlen(string)));
	SelectObject(_hDC, oldFont);

	if (screenID != ID_SCREEN)
		ReleaseDC(screenID, _hDC);
	else
		DeleteDC(_hDC);

	// Return the text height.
	return (height);
}

int UI_MSWINDOWS_DISPLAY::TextWidth(const char *string, SCREENID screenID,
	LOGICAL_FONT logicalFont)
{
	logicalFont &= 0x0FFF;
	HDC _hDC;
	if (screenID != ID_SCREEN)
		_hDC = GetDC(screenID);
	else
	{
		HDC tHDC = GetDC(GetDesktopWindow());
		_hDC = CreateCompatibleDC(tHDC);
		ReleaseDC(GetDesktopWindow(), tHDC);
	}

	// Get the string width.
	HFONT oldFont = SelectObject(_hDC, fontTable[logicalFont]);
	int width = LOWORD(GetTextExtent(_hDC, (LPSTR)string, ui_strlen(string)));
	SelectObject(_hDC, oldFont);

	if (screenID != ID_SCREEN)
		ReleaseDC(screenID, _hDC);
	else
		DeleteDC(_hDC);

	// Return the text width.
	return (width);
}

int UI_MSWINDOWS_DISPLAY::VirtualGet(SCREENID screenID, int, int, int, int)
{
	if (screenID)
		SendMessage(screenID, WM_SETREDRAW, FALSE, 0);
	return (TRUE);
}

int UI_MSWINDOWS_DISPLAY::VirtualPut(SCREENID screenID)
{
	if (screenID)
	{
		SendMessage(screenID, WM_SETREDRAW, TRUE, 0);
		ShowWindow(screenID, SW_SHOWNA);
	}
	return (TRUE);
}
