//	HLPBAR.CPP (HELPBAR) - This file contains the help bar class.
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <string.h>
#include <ui_win.hpp>
#include "hlpbar.hpp"

#ifdef _WINDOWS
static int _helpbarOffset = -1;
static FARPROC _helpbarCallback = (FARPROC)DefWindowProc;
static FARPROC _helpbarJumpInstance = NULL;

long FAR PASCAL _export HelpbarJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	HELP_BAR *object = (HELP_BAR *)GetWindowLong(hWnd, _helpbarOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}
#endif
HELP_BAR::HELP_BAR(char *_text) :
	UI_WINDOW_OBJECT(0, 0, 0, 0, WOF_NON_SELECTABLE | WOF_NON_FIELD_REGION | WOF_BORDER, WOAF_NO_FLAGS)
{
	searchID = windowID[0] = ID_HELP_BAR;
	StringID("HELP_BAR");

	// Initialize the help bar information.
	text = _text ? strdup(_text) : NULL;
}

HELP_BAR::~HELP_BAR(void)
{
	if (text)
		delete text;
}

#if defined(_WINDOWS) | defined(_Windows)
EVENT_TYPE HELP_BAR::Event(const UI_EVENT &event)
{
	const int HELP_OFFSET = 1;

	// Switch on the event type.
	EVENT_TYPE ccode = event.type;
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!_helpbarJumpInstance)
			_helpbarJumpInstance = (FARPROC)HelpbarJumpProcedure;
		UI_WINDOW_OBJECT::Event(event);
		break;

	case S_SIZE:
	case S_CREATE:
		UI_WINDOW_OBJECT::Event(event);
		true.left--; true.right++;
		true.top = ++true.bottom - display->cellHeight - 1;
		if (ccode == S_CREATE)
			RegisterObject("HELP_BAR", "STATIC", &_helpbarOffset,
				&_helpbarJumpInstance, &_helpbarCallback, NULL);
		break;

	default:
		WORD message = event.message.message;
		if ((ccode == S_REDISPLAY && screenID) ||
			(event.type == E_MSWINDOWS && message == WM_PAINT))
		{
			if (ccode == S_REDISPLAY)
				InvalidateRect(screenID, NULL, FALSE);
			PAINTSTRUCT ps;
			HDC hDC = BeginPaint(screenID, &ps);
			RECT region;
			GetClientRect(screenID, &region);

			// Fill the background.
			HBRUSH fillBrush = CreateSolidBrush(RGB_LIGHTGRAY);
			FillRect(hDC, &region, fillBrush);
			DeleteObject(fillBrush);

			// Draw the shadow.
			region.left += display->cellWidth;
			region.top += HELP_OFFSET;
			region.right -= display->cellWidth;		
			region.bottom -= HELP_OFFSET;
			HPEN darkShadow = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_BTNSHADOW));
			HPEN oldPen = SelectObject(hDC, darkShadow);
			MoveTo(hDC, region.left, region.bottom - 1);
			LineTo(hDC, region.left, region.top);
			LineTo(hDC, region.right, region.top);
			SelectObject(hDC, oldPen);
			DeleteObject(darkShadow);
			HPEN lightShadow = GetStockObject(WHITE_PEN);
			oldPen = SelectObject(hDC, lightShadow);
			LineTo(hDC, region.right, region.bottom);
			LineTo(hDC, region.left - 1, region.bottom);
			SelectObject(hDC, oldPen);

			// Draw the text.
			region.left += HELP_OFFSET; region.top++;
			region.right -= HELP_OFFSET; region.bottom--;
			COLORREF oldForeground = SetTextColor(hDC, RGB_BLACK);
			int oldMode = SetBkMode(hDC, TRANSPARENT);
			::DrawText(hDC, (LPSTR)text, ui_strlen(text), &region,
				DT_SINGLELINE | DT_VCENTER | DT_LEFT);
			SetBkMode(hDC, oldMode);
			SetTextColor(hDC, oldForeground);
			
			EndPaint(screenID, &ps);
		}
		else if (event.type == E_MSWINDOWS && message == WM_ERASEBKGND)
			return (TRUE);
		else
			ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#else
EVENT_TYPE HELP_BAR::Event(const UI_EVENT &event)
{
	const int HELP_OFFSET = 1;

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event);
	switch (ccode)
	{
	case S_CREATE:
		UI_WINDOW_OBJECT::Event(event);
		if (display->isText)
			true.top = true.bottom;
		else
		{
			true.left--; true.right++;
			true.top = ++true.bottom - display->cellHeight + 1;
		}
		break;

	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		UI_WINDOW_OBJECT::Event(event);
		if (FlagSet(woStatus, WOS_REDISPLAY) && display->isText)
		{
			UI_REGION region = true;
			UI_PALETTE *palette = LogicalPalette(ccode, ID_BUTTON);
			DrawText(screenID, region, text, palette, TRUE, ccode);
		}
		else if (FlagSet(woStatus, WOS_REDISPLAY))
		{
			UI_REGION region = true;
			if (FlagSet(woFlags, WOF_BORDER))
				DrawBorder(screenID, region, FALSE, ccode);
			UI_PALETTE *palette = LogicalPalette(ccode, ID_BUTTON);
			display->Rectangle(screenID, region, palette, 0, TRUE, FALSE, &clip);
			region.left += display->cellWidth;
			region.top += HELP_OFFSET;
			region.right -= display->cellWidth;		
			region.bottom -= (HELP_OFFSET + 1);
			palette = LogicalPalette(ccode, ID_DARK_SHADOW);
			display->Line(screenID, region.left, region.bottom - 1,
				region.left, region.top, palette, 1, FALSE, &clip);
			display->Line(screenID, region.left, region.top,
				region.right - 1, region.top, palette, 1, FALSE, &clip);
			palette = LogicalPalette(ccode, ID_WHITE_SHADOW);
			display->Line(screenID, region.right, region.top,
				region.right, region.bottom, palette, 1, FALSE, &clip);
			display->Line(screenID, region.right, region.bottom,
				region.left, region.bottom, palette, 1, FALSE, &clip);
			region.left += HELP_OFFSET; region.top++;
			region.right -= HELP_OFFSET; region.bottom--;
			palette = LogicalPalette(ccode, ID_BUTTON);
			DrawText(screenID, region, text, palette, TRUE, ccode);
			woStatus &= ~WOS_REDISPLAY;
		}
		break;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#endif

void *HELP_BAR::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	switch (request)
	{
	case GET_TEXT:
		if (!data)
			return (text);
		*(char **)data = text;
		break;

	case SET_TEXT:
		if (text)
			delete text;
		text = data ? strdup((char *)data) : NULL;
		HELP_BAR::Event(UI_EVENT(S_REDISPLAY));
		break;

	default:
		data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}
