//	GRAPH.CPP (GRAPH) - Extended graphics example.
//	COPYRIGHT (C) 1990-1993.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/

//  May be freely copied, used and distributed.

#include <stdlib.h>
#include <ui_win.hpp>

// Converts MiniCells to pixels
#define MiniConvertY(y) ((int)((y) * display->cellHeight * display->miniNumeratorY / display->miniDenominatorY))
#define MiniConvertX(x) ((int)((x) * display->cellWidth * display->miniNumeratorX / display->miniDenominatorX))

// Palettes used for drawing graphics.
UI_PALETTE graphPalette[] =
{
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_INTERLEAVE_FILL, BLACK, RED, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260',attrib( BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_INTERLEAVE_FILL, BLACK, YELLOW, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_INTERLEAVE_FILL, BLACK, GREEN, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_INTERLEAVE_FILL, BLACK, BLUE, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_INTERLEAVE_FILL, BLACK, LIGHTRED, BW_BLACK, BW_WHITE,	GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_SOLID_FILL, BLACK, LIGHTGREEN, BW_BLACK, BW_WHITE, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_SOLID_FILL, BLACK, DARKGRAY, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_SOLID_FILL, RED, DARKGRAY, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_SOLID_FILL, GREEN, DARKGRAY, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY }
};

UI_PALETTE textPalette =
{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
	PTN_INTERLEAVE_FILL, BLACK, RED, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY
};

class LINE_CHART : public UIW_WINDOW
{
public:
	LINE_CHART(int left, int top, int width, int height, WOF_FLAGS woFlags = WOF_BORDER);

	virtual EVENT_TYPE DrawItem(const UI_EVENT &event, EVENT_TYPE ccode);
};

LINE_CHART::LINE_CHART(int left, int top, int width, int height, WOF_FLAGS woFlags)
	: UIW_WINDOW(left, top, width, height, woFlags, WOAF_NO_FLAGS)
{
	woStatus |= WOS_OWNERDRAW;
}

EVENT_TYPE LINE_CHART::DrawItem(const UI_EVENT &event, EVENT_TYPE ccode)
{
	display->VirtualGet(screenID, true);

	UIW_WINDOW::DrawItem(event, ccode);

	// Initialize data for line graph.
	static int redLine[] = { 25, 20, 37, 48, 86, 79, 93, 0 };
	static int greenLine[] = { 20, 30, 35, 58, 50, 65, 80, 0 };
	const int left = true.left,
		bottom = true.bottom,
		width = 4 * display->cellWidth;

	// Create the graph lines for lineWindow.
	for (int x = 0; redLine[x+1]; x++)
		display->Line(screenID, left + x * width, bottom - MiniConvertY(redLine[x]),
			left + (x + 1) * width, bottom - MiniConvertY(redLine[x+1]),
			&graphPalette[7], 1, FALSE, &clip);

	for (x = 0; greenLine[x+1]; x++)
		display->Line(screenID, left + x * width, bottom - MiniConvertY(greenLine[x]),
			left + (x + 1) * width, bottom - MiniConvertY(greenLine[x+1]),
			&graphPalette[8], 1, FALSE, &clip);

	// Draw the associated text.
	static char *month[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", NULL };
	for (x = 0; month[x]; x++)
		display->Text(screenID, left + x * width + 1, bottom - 
			display->cellHeight + display->preSpace + display->postSpace,
			month[x], &textPalette, -1, FALSE, FALSE< &clip);

	display->VirtualPut(screenID);
	return (TRUE);
}

class BAR_CHART : public UIW_WINDOW
{
public:
	BAR_CHART(int left, int top, int width, int height, WOF_FLAGS woFlags);

	virtual EVENT_TYPE DrawItem(const UI_EVENT &event, EVENT_TYPE ccode);
};

BAR_CHART::BAR_CHART(int left, int top, int width, int height, WOF_FLAGS woFlags)
	: UIW_WINDOW(left, top, width, height, woFlags, WOAF_NO_FLAGS)
{
	woStatus |= WOS_OWNERDRAW;
}

EVENT_TYPE BAR_CHART::DrawItem(const UI_EVENT &event, EVENT_TYPE ccode)
{
	display->VirtualGet(screenID, true);

	UIW_WINDOW::DrawItem(event, ccode);

	static int heights[] = { 67, 46, 88, 81, 65, 32, 0 };
	static char *years[] =
		{ "1985", "1986", "1987", "1988", "1989", "1990", NULL };
	const int bottom = true.top + MiniConvertY(100),
		width = 4 * display->cellWidth;
	int left = 2 * display->cellWidth + true.left;

	for (int bar = 0; heights[bar]; bar++)
	{
		display->Rectangle(screenID, left, bottom - MiniConvertY(heights[bar]),
			left + width, bottom, &graphPalette[bar], 1, TRUE, FALSE, &clip);
		display->Text(screenID, left, bottom + display->cellHeight, years[bar],
			&graphPalette[6], -1, FALSE, FALSE, &clip);
		left += width + 15;
	}

	display->VirtualPut(screenID);
	return (TRUE);
}

class PIE_CHART : public UIW_WINDOW
{
public:
	PIE_CHART(int left, int top, int width, int height,
		WOF_FLAGS flags = WOF_NO_FLAGS);

	virtual EVENT_TYPE DrawItem(const UI_EVENT &event, EVENT_TYPE ccode);
};

PIE_CHART::PIE_CHART(int left, int top, int width, int height, WOF_FLAGS flags)
	: UIW_WINDOW(left, top, width, height, flags, WOAF_NO_FLAGS)
{
	woStatus |= WOS_OWNERDRAW;
}

EVENT_TYPE PIE_CHART::DrawItem(const UI_EVENT &event, EVENT_TYPE ccode)
{
	display->VirtualGet(screenID, true);

	UIW_WINDOW::DrawItem(event, ccode);

	static int sectors[] = { 0, 56, 112, 133, 210, 240, 360, 0 };

	// The following values are dependant on the above values.
	//  cosX[i] = 100 * cos((sector[i] + sector[i + 1]) / 2);
	//  cosX[i] = 100 * sin((sector[i] + sector[i + 1]) / 2);
	static int cosX[] = { 88, 10, -54, -99, -71,  50 };
	static int sinY[] = { 47, 99,  84,  15, -71, -87 };

	static char *label[] = { "Rent", "Food", "Clothing", "Auto", "Other", "Entertainment" };
	UI_POSITION center;
	center.column = true.Width() / 2;
	center.line = true.Height() / 2;

	display->Ellipse(screenID, true.left + center.column,
		true.top + center.line + display->cellHeight, 0, 360,
		center.column/2, center.line/2, &graphPalette[6], TRUE, FALSE, &clip);

	for (int sector = 0; sectors[sector+1]; sector++)
	{
		display->Ellipse(screenID, true.left + center.column, true.top + center.line,
			sectors[sector], sectors[sector+1], center.column/2, center.line/2,
			&graphPalette[sector], TRUE, FALSE, &clip);
		int left = cosX[sector] * center.column / 2 / 100;
		left += left > 0 ? display->cellWidth + true.left + center.column:
			-(display->TextWidth(label[sector], screenID, font) + display->cellWidth)
			+ true.left + center.column;
		int top = sinY[sector] * center.line / 2 / 100;
		top = top > 0 ? true.top + center.line - top - display->TextHeight(label[sector], screenID, font) :
			true.top + center.line - top;
		display->Text(screenID, left, top, label[sector],
			&graphPalette[sector], -1, FALSE, FALSE, &clip, font);
	}

	display->VirtualPut(screenID);
	return (TRUE);
}

#if defined(ZIL_MSDOS)
main()
{
	// Create the MSDOS display.
	UI_DISPLAY *display = new UI_GRAPHICS_DISPLAY;
#elif defined(ZIL_MSWINDOWS)
int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR, int nCmdShow)
{
	// Create the Windows display.
	UI_DISPLAY *display = new UI_MSWINDOWS_DISPLAY(hInstance, hPrevInstance, nCmdShow);
#elif defined(ZIL_OS2)
main()
{
	// Create the OS/2 display.
	UI_DISPLAY *display = new UI_OS2_DISPLAY;
#elif defined(ZIL_MOTIF)
main(int argc, char **argv)
{
	// Create the Motif display and initialize XtIntrinsics using application
	// class "ZincApp".
	UI_DISPLAY *display = new UI_MOTIF_DISPLAY(&argc, argv, "ZincApp");
#endif

	// Make sure the display installed correctly.
	if (!display || !display->installed)
	{
		delete display;
		return (0);
	}

	// Create the event manager and add devices.
	UI_EVENT_MANAGER *eventManager = new UI_EVENT_MANAGER(display);
	*eventManager
		+ new UID_KEYBOARD
		+ new UID_MOUSE
		+ new UID_CURSOR;

	// Create the window manager.
	UI_WINDOW_MANAGER *windowManager = new UI_WINDOW_MANAGER(display, eventManager);

	// Add the windows to the window manager.
	*windowManager
		+ &(*UIW_WINDOW::Generic(25, 3, 45, 16, "BAR")
			+ new BAR_CHART(20, 10, 390, 100 + display->cellHeight,
				WOF_MINICELL | WOF_BORDER))
		+ &(*UIW_WINDOW::Generic(15, 12, 60, 8, "SIZABLE PIE CHART")
			+ new PIE_CHART(0, 0, 0, 0, WOF_NON_FIELD_REGION))
		+ &(*UIW_WINDOW::Generic(34, 7, 52, 8, "PIE CHART")
			+ new PIE_CHART(0, 0, 50, 7))
		+ &(*UIW_WINDOW::Generic(7, 2, 30, 14, "LINE CHART")
			+ new LINE_CHART(20, 10, 240, 100, WOF_MINICELL | WOF_BORDER));

	// Wait for user response.
	UI_EVENT event;
	EVENT_TYPE ccode;
	do
	{
		eventManager->Get(event, Q_NORMAL);
		ccode = windowManager->Event(event);
	} while (ccode != L_EXIT && ccode != S_NO_OBJECT);

	// Clean up.
	delete windowManager;
	delete eventManager;
	delete display;

	return (0);
}
