//	Zinc Interface Library Designer - Z_IMAGE2.CPP
//	COPYRIGHT (C) 1990-1993.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


// Current limitations:
// 1 - Only 16 colors.
// 2 - No RLE

#include <io.h>
#include <fcntl.h>
#include <sys\stat.h>
#include <stdio.h>
#include <string.h>
#include <ui_gen.hpp>

#if !defined(MAX_PATH)
#define MAX_PATH	128
#endif

#if defined(ZIL_MSDOS) || defined(ZIL_OS2)
#define BI_RGB	0
#define BYTE	UCHAR
#define WORD	USHORT
#define DWORD	ULONG

struct RGBQUAD
{
	BYTE	rgbBlue;
	BYTE	rgbGreen;
	BYTE	rgbRed;
	BYTE	rgbReserved;
};

struct BITMAPFILEHEADER
{
    WORD	bfType;
    DWORD	bfSize;
    WORD	bfReserved1;
    WORD	bfReserved2;
    DWORD	offBits;
};

struct BITMAPINFOHEADER
{
	DWORD	biSize;
	DWORD	biWidth;
	DWORD	biHeight;
	WORD	biPlanes;
	WORD	biBitCount;
	DWORD	biCompression;
	DWORD	biSizeImage;
	DWORD	biXPelsPerMeter;
	DWORD	biYPelsPerMeter;
	DWORD	biClrUsed;
	DWORD	biClrImportant;
};

struct BITMAPINFO
{
  	BITMAPINFOHEADER	bmiHeader;
	RGBQUAD				bmiColors[1];
};

#endif

struct ICO_HEADER
{
	WORD	icoReserved;
	WORD	icoResourceType;
	WORD	icoResourceCount;
};

struct ICO_DESCRIPTOR
{
	BYTE	width;
	BYTE	height;
	BYTE	colorCount;
	BYTE	reserved1;
	WORD	reserved2;
	WORD	reserved3;
	DWORD	icoDIBSize;
	DWORD	icoDIBOffset;
};

// Map for matching RGB values to an index into _colorMap[16].
static RGBQUAD zincRGBValues[] =
{
	{ 0x00, 0x00, 0x00 },   // 0 - BLACK		
	{ 0x80, 0x00, 0x00 },	// 1 - BLUE		
	{ 0x00, 0x80, 0x00 },	// 2 - GREEN		
	{ 0x80, 0x80, 0x00 },	// 3 - CYAN		
	{ 0x00, 0x00, 0x80 },	// 4 - RED			
	{ 0x80, 0x00, 0x80 },	// 5 - MAGENTA		
	{ 0x00, 0x80, 0x80 },	// 6 - BROWN		
	{ 0xC0, 0xC0, 0xC0 },	// 7 - LIGHTGRAY	
	{ 0x80, 0x80, 0x80 },	// 8 - DARKGRAY	
	{ 0xFF, 0x00, 0x00 },	// 9 - LIGHTBLUE	
	{ 0x00, 0xFF, 0x00 },	// 10 - LIGHTGREEN	
	{ 0xFF, 0xFF, 0x00 },	// 11 - LIGHTCYAN	
	{ 0x00, 0x00, 0xFF },	// 12 - LIGHTRED	
	{ 0xFF, 0x00, 0xFF },	// 13 - LIGHTMAGENTA
	{ 0x00, 0xFF, 0xFF },	// 14 - YELLOW		
	{ 0xFF, 0xFF, 0xFF }	// 15 - WHITE		
};

UCHAR GetNearestZincColor(RGBQUAD rgbColor)
{
    for (UCHAR colorIndex = 0; colorIndex < 16 &&
        (zincRGBValues[colorIndex].rgbRed != rgbColor.rgbRed ||
		zincRGBValues[colorIndex].rgbGreen != rgbColor.rgbGreen ||
		zincRGBValues[colorIndex].rgbBlue != rgbColor.rgbBlue);
		colorIndex++)
    ;
    if (colorIndex == 16)
        colorIndex = 0xFF;
    return colorIndex;
}

#define		BITMAP_OK					0
#define		BITMAP_INVALID_FILE			1
#define		BITMAP_INVALID_FILE_TYPE	2
#define		BITMAP_TOO_LARGE			3
#define		BITMAP_COMPRESSED			4

int main(int argv, char *argc[])
{
	if (argv < 3)
	{
		printf("\n\tSyntax:  ico2dat <filename[.ico]> <filename[.dat]> [imagename]\n");
		return 1;
	}

	char sourcePath[MAX_PATH];
	strcpy(sourcePath, argc[1]);
	if (!strchr(sourcePath, '.'))
		strcat(sourcePath, ".bmp");

	char destPath[MAX_PATH];
	strcpy(destPath, argc[2]);
	if (!strchr(destPath, '.'))
		strcat(destPath, ".dat");

	char imageName[64];
	if (argv >= 4)
		strcpy(imageName, argc[3]);
	else
	{
		UI_STORAGE::StripFullPath(sourcePath, NULL, imageName);
		*strchr(imageName, '.') = '\0';
	}
	ui_strupr(imageName);

	USHORT colorWidth, colorHeight;
	UCHAR *colorArray = NULL;

	ICO_HEADER header;
	ICO_DESCRIPTOR descriptor;
	RGBQUAD rgbColor;
	UCHAR zincRGBIndex[16] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	UCHAR *zincBitmap;

	int fileHandle;
	BYTE *xorArray;
	BYTE *andArray;
	int j, k;
	int XORSize, ANDSize;
	int icoIndex;
	int zincIndex;

	if ((fileHandle = open(sourcePath, O_RDONLY | O_BINARY)) == -1)
	{
		printf("Error opening source file: %s\n", sourcePath);
		return 1;
	}

	// Check icon file header.
	read(fileHandle, &header, sizeof(header));
	if (header.icoReserved != 0 || header.icoResourceType != 1)
	{
		printf("\"%s\" is not a Windows icon file.", sourcePath);
		return 1;
	}

	// Read first icon descriptor.
	lseek(fileHandle, sizeof(header), SEEK_SET);
	read(fileHandle, &descriptor, sizeof(descriptor));

	if (descriptor.colorCount == 16 && descriptor.width == 32 && descriptor.height == 32)
	{
		lseek(fileHandle, descriptor.icoDIBOffset + sizeof(BITMAPINFOHEADER), SEEK_SET);

		// Match up the palette maps.
		for (j = 0; j < descriptor.colorCount; j++)
		{
			read(fileHandle, &rgbColor, sizeof(RGBQUAD));
			for (k = 0; k < descriptor.colorCount; k++)
				if (zincRGBValues[k].rgbRed == rgbColor.rgbRed &&
					zincRGBValues[k].rgbGreen == rgbColor.rgbGreen &&
					zincRGBValues[k].rgbBlue == rgbColor.rgbBlue)
				{
					zincRGBIndex[j] = k;
					break;
				}
		}

		// Read the ICO format arrays from disk.
		XORSize = 512;
		ANDSize = 128;

		xorArray = new BYTE [XORSize];
		zincBitmap = new UCHAR [descriptor.width * descriptor.height];
		read(fileHandle, xorArray, XORSize);
		andArray = new BYTE [ANDSize];
		read(fileHandle, andArray, ANDSize);

		// Convert the ICO format array to Zinc format icon.
		zincIndex = 0;
		for (j = 31; j >= 0; j--)
		{
			for (k = 0; k < 16; k++)
			{
				// Map the corresponding color into the bitmap array.
				icoIndex = j * 16 + k;
				zincBitmap[zincIndex++] = zincRGBIndex[(xorArray[icoIndex] >> 4) & 0xF];
				zincBitmap[zincIndex++] = zincRGBIndex[xorArray[icoIndex] & 0xF];
			
				// If the pixel is transparent (AND array) then set to white.
				if (andArray[(j * 4 + k / 4)] & (0x80 >> ((k % 4) * 2)))
					zincBitmap[zincIndex - 2] = 15;
				if (andArray[(j * 4 + k / 4)] & (0x80 >> ((k % 4) * 2 + 1)))
					zincBitmap[zincIndex - 1] = 15;
			}
		}
		colorWidth = descriptor.width;
		colorHeight = descriptor.height;
		colorArray = zincBitmap;
	}
	else
	{
		printf("File \"%s\" is not of a compatible format.", sourcePath);
		return 1;
	}
	close(fileHandle);

	UI_STORAGE *storage = new UI_STORAGE(destPath, UIS_READWRITE);
	if (storage->storageError)
	{
		delete storage;
		storage = new UI_STORAGE(destPath, UIS_OPENCREATE);
		if (storage->storageError)
		{
			printf("\nError (%d) opening output file: %s", storage->storageError, destPath);
			return 1;
		}
		storage->MkDir("UIW_WINDOW");
		storage->MkDir("UI_BITMAP");
		storage->MkDir("UI_ICON");
		storage->MkDir("UI_HELP");
	}

	storage->ChDir("~UI_ICON");
	{
	UI_STORAGE_OBJECT sObject(*storage, imageName, 0, UIS_READWRITE | UIS_CREATE);
	sObject.Store(colorWidth);
	sObject.Store(colorHeight);
	sObject.Store(colorArray, colorWidth, colorHeight);
	}
	delete colorArray;

	storage->Save();
	delete storage;

 	return 0;
}
