//	Zinc Interface Library Designer - W_BITMAP.CPP
//	COPYRIGHT (C) 1990-1993.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "ui_dsn.hpp"
#if defined(_MSC_VER)
#pragma hdrstop					// Microsoft pre-compiled header pragma.
#endif

// ----- Definition for the Windows UIW_BITMAP callback function -----
static int _bitmapOffset = -1;
#if defined(WIN32)
static WNDPROC _bitmapCallback = NULL;
#else
static FARPROC _bitmapCallback = (FARPROC)DefWindowProc;
static FARPROC _bitmapJumpInstance = NULL;

long FAR PASCAL _export BitmapJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)GetWindowLong(hWnd, _bitmapOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}
#endif

EVENT_TYPE BITMAP_OBJECT::DrawItem(const UI_EVENT &event, EVENT_TYPE ccode)
{
	PAINTSTRUCT ps;
	BeginPaint(screenID, &ps);

	HBRUSH brush = CreateSolidBrush(RGB_BLACK);
	FrameRect(ps.hdc, &ps.rcPaint, brush);
	DeleteObject(brush);

	int imageWidth = width, imageHeight = height;
	if (FlagSet(userFlags, BMF_VIEW_BITMAP))
	{
		imageWidth += 16;
		imageHeight += 16;
	}

	HBITMAP colorBitmap;
	int i = 0, j = 0;
	UCHAR *tempArray = new UCHAR[imageWidth * imageHeight];
	for (int y = 0; y < imageHeight; y++)
	{
		for (int x = 0; x < imageWidth; x++)
		{
			if ((FlagSet(userFlags, BMF_VIEW_BITMAP) && (x < 8 || y < 8 || x > width + 7 || y > height + 7))
				|| bitmapArray[j++] == 0xFF)
				tempArray[i++] = editor->screenColor;
			else
				tempArray[i++] = bitmapArray[j - 1];
		}
	}
	display->BitmapArrayToHandle(screenID, imageWidth, imageHeight,
		tempArray, NULL, &colorBitmap, NULL);
	delete tempArray;

	HDC hMemDC = CreateCompatibleDC(ps.hdc);
	SelectObject(hMemDC, colorBitmap);
	StretchBlt(ps.hdc, 1, 1, imageWidth * ratioWidth, imageHeight * ratioHeight, hMemDC, 0, 0, imageWidth, imageHeight, SRCCOPY);
	DeleteDC(hMemDC);
	DeleteObject(colorBitmap);

	if (FlagSet(userFlags, BMF_GRID))
	{
		HPEN hPen = CreatePen(PS_SOLID, 1, RGB_BLACK);
		HPEN oldPen = SelectObject(ps.hdc, hPen);
		for (int i = 1; i <= width; i++)
		{
#if !defined(WIN32)
			MoveTo(ps.hdc, i * ratioWidth, 0);
#else
			MoveToEx(ps.hdc, i * ratioWidth, 0, 0);
#endif
			LineTo(ps.hdc, i * ratioWidth, height * ratioHeight);
		}
		for (i = 1; i <= height; i++)
		{
#if !defined(WIN32)
			MoveTo(ps.hdc, 0, i * ratioHeight);
#else
			MoveToEx(ps.hdc, 0, i * ratioHeight, 0);
#endif
			LineTo(ps.hdc, width * ratioWidth, i * ratioHeight);
		}
		SelectObject(ps.hdc, oldPen);
		DeleteObject(hPen);
	}

	EndPaint(screenID, &ps);

	return TRUE;
}

void BITMAP_OBJECT::Pixel(int column, int row, int color)
{
	int index = row * width + column;
	if (bitmapArray[index] != color)
	{
		if (FlagSet(userFlags, BMF_VIEW_BITMAP))
		{
			column += 9;
			row += 9;
			HDC hDC = GetDC(screenID);
			SetPixel(hDC, column, row, display->colorMap[(int)(color == 0xFF ?
				editor->screenColor : color)].colorForeground);
			ReleaseDC(screenID, hDC);
		}
		else
		{
			int grid = FlagSet(userFlags, BMF_GRID) ? 2 : 1;

			RECT rect;
			rect.left = column * ratioWidth + 1;
			rect.top = row * ratioHeight + 1;
			rect.right = rect.left + ratioWidth - grid + 1;
			rect.bottom = rect.top + ratioHeight - grid + 1;

			HDC hDC = GetDC(screenID);
			HBRUSH brush = CreateSolidBrush(display->colorMap[(int)(color == 0xFF ?
				editor->screenColor : color)].colorForeground);
			FillRect(hDC, &rect, brush);
			DeleteObject(brush);
			ReleaseDC(screenID, hDC);
		}

		if (FlagSet(userFlags, BMF_EDIT_BITMAP))
			editor->viewBitmap->Pixel(column, row, color);
		else
			bitmapArray[index] = color;
	}
}

EVENT_TYPE BITMAP_OBJECT::Event(const UI_EVENT &_event)
{
	UI_EVENT event = _event;
	if (event.type == E_MSWINDOWS)
	{
		if (event.message.hwnd != screenID)
			return S_UNKNOWN;
		WORD message = event.message.message;
		WORD wParam = event.message.wParam;
		if (message == WM_LBUTTONDOWN || message == WM_RBUTTONDOWN ||
			(message == WM_MOUSEMOVE && (wParam == MK_LBUTTON || wParam == MK_RBUTTON)))
		{
//			event.type = E_MOUSE;
			event.type = (message == WM_LBUTTONDOWN || message == WM_RBUTTONDOWN) ? L_BEGIN_SELECT : L_CONTINUE_SELECT;
			if (message == WM_LBUTTONDOWN || wParam == MK_LBUTTON)
				event.rawCode = M_LEFT;
			else if (message == WM_RBUTTONDOWN || wParam == MK_RBUTTON)
				event.rawCode = M_RIGHT;
			event.position.column += true.left;
			event.position.line += true.top;
		}
		else if (message == WM_PAINT)
		{
			DrawItem(event, event.type);
		}
	}

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW_OBJECT);
	switch (ccode)
	{
	case S_INITIALIZE:
#if !defined(WIN32)
		if (!_bitmapJumpInstance)
			_bitmapJumpInstance = (FARPROC)BitmapJumpProcedure;
#endif
		dwStyle |= BS_OWNERDRAW;
		UI_WINDOW_OBJECT::Event(event);
		break;

	case S_CREATE:
#if defined(WIN32)
		RegisterObject("UIW_BITMAP", NULL, &_bitmapCallback, NULL);
#else
		RegisterObject("UIW_BITMAP", NULL, &_bitmapOffset,
			&_bitmapJumpInstance, &_bitmapCallback, NULL);
#endif
		break;

	case S_SIZE:
		if (FlagSet(userFlags, BMF_EDIT_BITMAP))
		{
			int tWidth = editor->client.right - editor->client.left - 21 * display->cellWidth - width - 16;
			int tHeight = editor->client.bottom - editor->client.top - 3 * display->cellHeight;
			ratioWidth = ratioHeight = Min(tWidth / width, tHeight / height);
			relative.right = relative.left + width * ratioWidth + 1;
			relative.bottom = relative.top + height * ratioHeight + 1;
		}
		else if (FlagSet(userFlags, BMF_VIEW_BITMAP))
		{
			relative.right = editor->client.right - editor->client.left - 3 * display->cellWidth;
			relative.left = relative.right - width - 17;
			relative.bottom = relative.top + height + 17;
		}
		else if (FlagSet(userFlags, BMF_LT_RT_BITMAP))
		{
			relative.right = editor->client.right - editor->client.left - 3 * display->cellWidth;
			relative.left = relative.right - width * ratioWidth - 1;
			relative.bottom = editor->client.bottom - editor->client.top - 4 * display->cellHeight;
			relative.top = relative.bottom - height * ratioHeight - 1;
		}
		else
		{
			relative.right = relative.left + width * ratioWidth + 1;
			relative.bottom = relative.top + height * ratioHeight + 1;
		}
		true = relative;
		UI_WINDOW_OBJECT::Event(event);

		// Save editor's client region.
		if (FlagSet(userFlags, BMF_COLOR_BITMAP))
		{
			RECT rect;
			GetClientRect(editor->screenID, &rect);
			editor->client.Assign(rect);
		}
		break;

	case L_BEGIN_SELECT:
		parent->Event(IMAGE_SAVE_UNDO);
		// Continue to L_CONTINUE_SELECT.

	case L_CONTINUE_SELECT:
		{
		int x = Min((event.position.column - true.left) / ratioWidth, width - 1);
		int y = Min((event.position.line - true.top) / ratioHeight, height - 1);
		int color = editor->leftRightColors[FlagSet(event.rawCode, M_LEFT) ? 0 : 1];
		if (FlagSet(userFlags, BMF_EDIT_BITMAP))
		{
			switch (editor->drawMode)
			{
			case IMAGE_PENCIL:
				{
				static int lastX, lastY;
				if (ccode == L_BEGIN_SELECT)
					Pixel(x, y, color);
				else
					Line(lastX, lastY, x, y, color);
				lastX = x;
				lastY = y;
				}
				break;

			case IMAGE_BRUSH:
				{
				UI_REGION rect;
				int brush = editor->brushSize;
				rect.left = Max(x - brush / 2, 0);
				rect.top = Max(y - brush / 2, 0);
				rect.right = Min(rect.left + brush - 1, width - 1);
				rect.bottom = Min(rect.top + brush - 1, height - 1);
				Rectangle(rect.left, rect.top, rect.right, rect.bottom, color, TRUE);
				}
				break;

			case IMAGE_LINE:
				{
					UI_POSITION p2 = GetRegion(x, y, IMAGE_LINE);
					Line(x, y, p2.column, p2.line, color);
				}
				break;

			case IMAGE_RECTANGLE:
			case IMAGE_FILL_RECTANGLE:
				{
					UI_POSITION p2 = GetRegion(x,y, IMAGE_RECTANGLE);
					int left = Min(x, p2.column);
					int top = Min(y, p2.line);
					int right = Max(x, p2.column);
					int bottom = Max(y, p2.line);
					Rectangle(left, top, right, bottom, color, (editor->drawMode == IMAGE_FILL_RECTANGLE) ? TRUE : FALSE);
				}
				break;

			case IMAGE_ELLIPSE:
			case IMAGE_FILL_ELLIPSE:
				{
					UI_POSITION p2 = GetRegion(x,y, IMAGE_RECTANGLE);
					int left = Min(x, p2.column);
					int top = Min(y, p2.line);
					int right = Max(x, p2.column);
					int bottom = Max(y, p2.line);
					Ellipse(left, top, right, bottom, color, (editor->drawMode == IMAGE_FILL_ELLIPSE) ? TRUE : FALSE);
				}
				break;

			case IMAGE_FILL:
				Fill(x, y, color);
				break;
			}
		}
		else if (FlagSet(userFlags, BMF_COLOR_BITMAP))
		{
			UCHAR color = bitmapArray[y * width + x];
			editor->leftRightBitmap->Pixel(FlagSet(event.rawCode, M_LEFT) ? 0 : 1, 0, color);
		}
		else if (FlagSet(userFlags, BMF_VIEW_BITMAP))
		{
			editor->screenBitmap->Pixel(0, 0, color);
			editor->leftRightBitmap->Event(S_REDISPLAY);
			editor->editBitmap->Event(S_REDISPLAY);
			UI_WINDOW_OBJECT::Event(S_REDISPLAY);
		}
		else if (FlagSet(userFlags, BMF_SCREEN_BITMAP))
			editor->leftRightBitmap->Pixel(FlagSet(event.rawCode, M_LEFT) ? 0 : 1, 0, 0xFF);
		}
		break;
 
	default:
  		ccode = UI_WINDOW_OBJECT::Event(event);

	}

	// Return the control code.
	return (ccode);
}

UI_POSITION BITMAP_OBJECT::GetRegion(int x, int y, EVENT_TYPE drawMode)
{
	UI_POSITION origin;
	UI_POSITION position;
	origin.column = position.column = x * ratioWidth + ratioWidth / 2 + 1;
	origin.line = position.line = y * ratioHeight + ratioHeight / 2 + 1;

	RECT region = { origin.column, origin.line, position.column, position.line };

	HDC hDC = GetDC(screenID);
	HBRUSH brush = CreateSolidBrush(RGB_WHITE);
	HPEN hPen = CreatePen(PS_SOLID, 1, RGB_WHITE);
	HPEN oldPen = SelectObject(hDC, hPen);
	int oldROP2 = SetROP2(hDC, R2_XORPEN);
	if (drawMode == IMAGE_LINE)
	{
#if !defined(WIN32)
		MoveTo(hDC, origin.column, origin.line);
#else
		MoveToEx(hDC, origin.column, origin.line, 0);
#endif
		LineTo(hDC, position.column, position.line);
	}
	else
		DrawFocusRect(hDC, &region);

	WORD message;
	EVENT_TYPE ccode = S_UNKNOWN;
	do
	{
		UI_EVENT event;
		eventManager->Get(event);
		message = event.message.message;
 		if (message == WM_LBUTTONDOWN || message == WM_NCLBUTTONDOWN ||
			(message == WM_MOUSEMOVE && FlagSet(event.message.wParam, MK_LBUTTON)))
		{
			if (event.position.column < 1)
				event.position.column = 1;
			else if (event.position.column > width * ratioHeight)
				event.position.column = width * ratioHeight;
			if (event.position.line < 1)
				event.position.line = 1;
			else if (event.position.line > height * ratioHeight)
				event.position.line = height * ratioHeight;
			UI_POSITION oldPosition = position;
			position = event.position;
			if (position != oldPosition)
			{
				if (drawMode == IMAGE_LINE)
				{
#if !defined(WIN32)
					MoveTo(hDC, origin.column, origin.line);
#else
					MoveToEx(hDC, origin.column, origin.line, 0);
#endif
					LineTo(hDC, oldPosition.column, oldPosition.line);
#if !defined(WIN32)
					MoveTo(hDC, origin.column, origin.line);
#else
					MoveToEx(hDC, origin.column, origin.line, 0);
#endif
					LineTo(hDC, position.column, position.line);
				}
				else
				{
					DrawFocusRect(hDC, &region);
					region.left = Min(position.column, origin.column);
					region.top = Min(position.line, origin.line);
					region.right = Max(position.column, origin.column);
					region.bottom = Max(position.line, origin.line);
					DrawFocusRect(hDC, &region);
				}
			}
		}
	} while (message != WM_LBUTTONUP && ccode != L_SELECT && ccode != L_CANCEL);
	if (drawMode == IMAGE_LINE)
	{
#if !defined(WIN32)
		MoveTo(hDC, origin.column, origin.line);
#else
		MoveToEx(hDC, origin.column, origin.line, 0);
#endif
		LineTo(hDC, position.column, position.line);
	}
	else
		DrawFocusRect(hDC, &region);
	SetROP2(hDC, oldROP2);
	SelectObject(hDC, oldPen);
	DeleteObject(hPen);
	DeleteObject(brush);
	ReleaseDC(screenID, hDC);

	UI_POSITION returnPosition;
	returnPosition.column = (position.column - 1) / ratioWidth;
	returnPosition.line = (position.line - 1) / ratioHeight;
	return returnPosition;
}

