//	Zinc Interface Library - I_BTCAT.CPP
//	COPYRIGHT (C) 1990-1993.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


// Warning!!!  geninterrupt() generates different code depending on
// whether the parameter is a small constant, large constant, or variable.
// This difference can cause some of this module to break, especially
// where ES and AX are used.  Change carefully.

#include <conio.h>
#include <signal.h>
#include <dos.h>
#include <mem.h>
#include <stdlib.h>
#ifdef DOSX286
#include <phapi.h>
#endif
#include "ui_evt.hpp"

// ---- UI_TEXT_DISPLAY ----
static int _originalMode = -1;

void I_ScreenOpen(int *mode, int *lines, int *columns)
{
	union REGS inregs, outregs;

	// Save original text mode.
	struct text_info textInfo;
	if (_originalMode == -1)
	{
		gettextinfo(&textInfo);
		_originalMode = textInfo.currmode;
	}

	// Initialize the text display.
	textmode(*mode);
	I_CursorRemove();

	// Check for special text mode monitors.
	gettextinfo(&textInfo);
	*mode = textInfo.currmode;
	*lines = textInfo.screenheight;
	*columns = textInfo.screenwidth;

	// Turn the blink attribute off.
	inregs.x.ax = 0x1003;
	inregs.h.bl = 0x00;
	int86(0x10, &inregs, &outregs);

	UCHAR newSetting = *((UCHAR _far *)MK_FP(0x40, 0x65));
	newSetting &= ~0x20;
	USHORT addr = *((USHORT _far *)MK_FP(0x40, 0x63)) + 4;
	outportb(addr, newSetting);
	// Update BIOS data area
	*((UCHAR _far *)MK_FP(0x40, 0x65)) = newSetting;
}

void I_ScreenClose(void)
{
	// Turn the blink attribute back on.
	union REGS inregs, outregs;
	inregs.x.ax = 0x1003;
	inregs.h.bl = 0x01;
	int86(0x10, &inregs, &outregs);

	_setcursortype(_NORMALCURSOR);
	textmode(_originalMode);
	clrscr();
}

void I_ScreenPut(int left, int top, int right, int bottom, void *buffer)
{
	puttext(left, top, right, bottom, buffer);
}

void I_ScreenGet(int left, int top, int right, int bottom, void *buffer)
{
	gettext(left, top, right, bottom, buffer);
}

int I_ScreenColor(int color)
{
	return (color & 0x0f);
}

void I_CursorRemove(void)
{
	_setcursortype(_NOCURSOR);
}

void I_CursorPosition(int y, int x, int val)
{
	gotoxy(x, y);
	_setcursortype((val == DC_INSERT) ? _SOLIDCURSOR : _NORMALCURSOR);
}

//		  Narrow, Wide
#if defined(OEM)
	ichar_t _iCornerUL[] = { 0xDA, 0xC9 };
	ichar_t _iCornerUR[] = { 0xBF, 0xBB };
	ichar_t _iCornerLL[] = { 0xC0, 0xC8 };
	ichar_t _iCornerLR[] = { 0xD9, 0xBC };
	ichar_t _iHorizontal[] = { 0xC4, 0xCD };
	ichar_t _iVertical[] = { 0xB3, 0xBA };
	ichar_t _iLowerBox = 0xdc;
	ichar_t _iUpperBox = 0xdf;
	ichar_t _iFullBox = 0xdb;
	ichar_t *_iSBMiddleArrow = "\xfe";
#else
	ichar_t _iCornerUL[] = { 0x8d, 0x9d };
	ichar_t _iCornerUR[] = { 0x85, 0x95 };
	ichar_t _iCornerLL[] = { 0x86, 0x96 };
	ichar_t _iCornerLR[] = { 0x8c, 0x9c };
	ichar_t _iHorizontal[] = { 0x8a, 0x9a };
	ichar_t _iVertical[] = { 0x83, 0x93 };
	ichar_t _iLowerBox = 0x91;
	ichar_t _iUpperBox = 0x90;
	ichar_t _iFullBox = 0x92;
	ichar_t *_iSBMiddleArrow = "\x9e";
#endif
ichar_t *_iSBUpArrow = "";
ichar_t *_iSBDownArrow = "";
ichar_t *_iSBLeftArrow = "";
ichar_t *_iSBRightArrow = "";
ichar_t *_iWDownArrow = "[]";
ichar_t *_iWNormalMin = "[]";
ichar_t *_iWMinimized = "[]";
ichar_t *_iWNormalMax = "[]";
ichar_t *_iWMaximized = "[]";
ichar_t *_iActiveCheck   = "[X] ";
ichar_t *_iInactiveCheck = "[ ] ";
ichar_t *_iActiveRadio   = "() ";
ichar_t *_iInactiveRadio = "( ) ";
ichar_t *_iCheckMark = "*";

// ---- UID_MOUSE ----
static unsigned _oldMouseMask = 0;
static unsigned char _oldMouseState = 0;
static unsigned _mouseState = 0;
static int _horizontalMickeys = 0;
static int _verticalMickeys = 0;

#define mouseInt	0x33
#define mouseInit	0
#define mouseUninit	0
#define mouseMickeys	11
#define mouseSetFunction	12
#define mouseExgFunction	20
static int _stopInterrupt = FALSE;

#pragma inline
static unsigned _oldMouseSegment = 0;
static unsigned _oldMouseOffset = 0;

static void MouseISR(void)
{
	// Get the proper value to ds.
	asm push ds
	asm mov ax, DGROUP
	asm mov ds, ax

#if defined(DOSX286)
	REGS16 regs;
#else
	union REGS regs;
#endif
	
	// Get the mouse information.
	if (!_stopInterrupt)
	{
		_stopInterrupt = TRUE;	// Prevent the interrupt from multiple calls.

		// Set up new mouse state. (Current state is in _BL)
		_mouseState = ((((_oldMouseState ^ _BL) << 4) | _BL) << 8) | (*((UCHAR _far *)MK_FP(0x40, 0x17)) & 0xF);
		_oldMouseState = _BL;

		// Compute the mouse movement.
#ifdef  DOSX286
		// We can't call int86(0x33,,) here; we blow up in the extender.  
		// Instead, we simply issue the interrupt in real mode and 
		// shuttle the results back up to protected mode.

		memset(&regs, 0, sizeof(REGS16));
		regs.ax = mouseMickeys;
		DosRealIntr(mouseInt, &regs, 0L, 0);
		_horizontalMickeys += regs.cx;
		_verticalMickeys += regs.dx;
#else
		regs.x.ax = mouseMickeys;
		int86(mouseInt, &regs, &regs);
		_horizontalMickeys += regs.x.cx;
		_verticalMickeys += regs.x.dx;
#endif
		// Set the internal mouse information for use in the Poll routine.
		if (mouseQueueEvent)
			mouseQueueEvent(_mouseState, &_horizontalMickeys, &_verticalMickeys);
		_stopInterrupt = FALSE;
	}

	// Restore the ds register.
	asm pop ds
}

int I_MouseOpen(void)
{
#if !defined(DOSX286)
	unsigned char far *far *mouseVector = (unsigned char far *far *) (4 * mouseInt);
	if (*mouseVector && **mouseVector != 0xCF)
	{
		_AX = mouseExgFunction;
		_BX = 0;
		_DX = 0;
		_CX = 0;
		geninterrupt(mouseInt);
		_BX = _ES;
		_oldMouseMask = _CX;
		_oldMouseSegment = _BX;
		_oldMouseOffset = _DX;
	}
#endif
	_AX = 0;
	geninterrupt(mouseInt);
	if (_AX != 0xFFFF)
		return FALSE;
	_CX = 0xFF;
	_DX = FP_OFF(MouseISR);
	_AX = FP_SEG(MouseISR);
	_ES = _AX;
	_AX = mouseSetFunction;
	geninterrupt(mouseInt);
	return TRUE;
}

void I_MouseClose(void)
{
#if !defined(DOSX286)
	_CX = _oldMouseMask;
	_DX = _oldMouseOffset;
	_BX = _oldMouseSegment;
	_ES = _BX;
	_AX = mouseExgFunction;
	geninterrupt(mouseInt);
#endif
	_AX = mouseUninit;
	geninterrupt(mouseInt);
}

void I_MouseHalt(int halt)
{
	_stopInterrupt = halt;
}

// ---- UID_KEYBOARD ----
static int _enhancedBios = 0;		// 0 or 0x10 (keyboard read)
static int _oldBreakState = 0;

static int CtrlCHandler(void)
{
	if (UID_KEYBOARD::breakHandlerSet == 0)
		exit(1);
	return 1;
}

void I_KeyboardOpen(void)
{
	union REGS inregs, outregs;

	// Check for enhaced keyboard by calling extended shift status function.
	inregs.x.ax = 0x12FF;
	int86(0x16, &inregs, &outregs);
	if (outregs.h.al != 0xFF)
	{
		_enhancedBios = 0x10;
		return;
	}
	// Double check for enhanced keyboard BIOS according to the
	// IBM PS/2 Technical Reference manual, page 2-103
	inregs.h.ah = 0x05;
	inregs.x.cx = 0xFFFF;
	int86(0x16, &inregs, &outregs);
	if (outregs.h.al != 0x00)
		return;
	// success, carry on, read that guy back ...
	inregs.h.ah = 0x10;
	for (int bufferIndex = 0; bufferIndex < 16; bufferIndex++)
	{
		int86(0x16, &inregs, &outregs);
		if (outregs.x.ax == 0xFFFF)
		{
			_enhancedBios = 0x10;
			break;
		}
	}
	// Set break interupt to ignore Control-Break/Control-C.
	ctrlbrk(CtrlCHandler);
	// Save previous state of Control-Break using INT 21H, 00H.
	inregs.x.ax = 0x3300;
	int86(0x21, &inregs, &outregs);
	_oldBreakState = outregs.h.dl;

	// Set Control-Break ON using INT 21H, 01H.
	inregs.x.ax = 0x3301;
	inregs.h.dl = 0x01;
	int86(0x21, &inregs, &outregs);
}

void I_KeyboardClose(void)
{
	union REGS inregs, outregs;

	// Set Control-Break status to previous status.
	inregs.x.ax = 0x3301;
	inregs.h.dl = _oldBreakState;
	int86(0x21, &inregs, &outregs);
}

void I_KeyboardRead(unsigned *rawCode, unsigned *shiftState, unsigned *value)
{
	union REGS inregs, outregs;

	inregs.h.ah = _enhancedBios + 0x00;
	int86(0x16, &inregs, &outregs);
	*rawCode = outregs.x.ax;
	*value = outregs.h.al;

	// Get the shift state using INT 16H, 12H(or 02H if not enhanced).
	inregs.h.ah = _enhancedBios + 0x02;
	int86(0x16, &inregs, &outregs);
	*shiftState = outregs.h.al;
	if (_enhancedBios && (outregs.x.ax & 0x0800) && *value != 0)
		*shiftState &= ~0x08;
}

int I_KeyboardQuery(unsigned *shiftState)
{
	union REGS inregs, outregs;

	// Test if keystroke waiting.
	inregs.h.ah = _enhancedBios + 0x01;
	int86(0x16, &inregs, &outregs);
	if (outregs.x.flags & 0x40)
	{
		// No keystroke waiting.
		// Check for ALT key using INT 16H, 02H.
		inregs.h.ah = _enhancedBios + 0x02;
		int86(0x16, &inregs, &outregs);
		*shiftState = outregs.h.al;
		return FALSE;
	}
	return TRUE;
}


