//	VLIST.HPP (VLIST) - This file contains source code for virtual list tutorial.
//	COPYRIGHT (C) 1990-1993.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/

//  May be freely copied, used and distributed.

#include <stdio.h>
#include <ui_win.hpp>
#include "vlist.hpp"

// The information file "vlist.txt" really contains fixed length records.
// The words are listed in alphabetical order and padded to 78 characters
// with spaces.  The file is formatted as a text file, so each operating 
// system has its own representation of the end-of-line.

#if defined(ZIL_UNIX)
#	define EOL_LENGTH	1
#elif defined(ZIL_MSDOS) || defined(ZIL_MSWINDOWS) || defined(ZIL_OS2)
#	define EOL_LENGTH	2
#endif

EVENT_TYPE VIRTUAL_ELEMENT::Event(const UI_EVENT &event)
{
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW);

	switch (ccode)
	{
	case S_INITIALIZE:
		ccode = UIW_STRING::Event(event);
		if (!display->isText)
			relative.bottom = relative.top + height;
		break;

	default:
		ccode = UIW_STRING::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

VIRTUAL_LIST::VIRTUAL_LIST(const char *fileName, int _recordLength) :
	UIW_WINDOW(0, 0, 0, 0, WOF_NON_FIELD_REGION), recordLength(_recordLength),
	topRecordShowing(0), numberShowing(0)
{
	// Open the database, get the total number of records.
	file = fopen(fileName, "rb");
	fseek(file, 0L, SEEK_END);
	numberOfRecords = (int)(ftell(file) / recordLength);
}

VIRTUAL_LIST::~VIRTUAL_LIST(void)
{
	// Close the database.
	if (file)
		fclose(file);
}

EVENT_TYPE VIRTUAL_LIST::Event(const UI_EVENT &event)
{
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW);

	switch (ccode)
	{
	case S_CREATE:
	case S_SIZE:
		{
		ccode = UIW_WINDOW::Event(event);

		// Calculate the number of elements that will fit in the list.
		int lineHeight = display->TextHeight("Mxq", screenID, font)
			+ display->preSpace + display->postSpace;

		int newNumberShowing = (true.bottom - true.top + display->preSpace + display->postSpace);
		newNumberShowing /= lineHeight;
		if (display->isText)
			newNumberShowing++;

		// Make sure the window is full of records.
		numberShowing = newNumberShowing;
		Destroy();
		int right = true.right - true.left + 1;
		for (int line = 0; line < numberShowing; line++)
		{
			VIRTUAL_ELEMENT *element = new VIRTUAL_ELEMENT(0,
				line * lineHeight, right,
				lineHeight, recordLength + 10);
			element->woStatus |= WOS_GRAPHICS;
			LoadRecord(element, topRecordShowing + line);
			Add(element);
		}
		Event(UI_EVENT(S_REDISPLAY, 0));
		}
		break;

	case L_PREVIOUS:
	case L_UP:
		if (Current() && Current()->Previous())
			Add(Current()->Previous());
		else if (topRecordShowing != 0 && First())
		{
			for (VIRTUAL_ELEMENT *object = Last(); object && object->Previous(); object = object->Previous())
				object->DataSet(object->Previous());
			LoadRecord(First(), --topRecordShowing);
		}
		break;

	case L_NEXT:
	case L_DOWN:
		if (Current() && Current()->Next())
			Add(Current()->Next());
		else if (topRecordShowing < numberOfRecords - 1 && First())
		{
			for (VIRTUAL_ELEMENT *object = First(); object && object->Next(); object = object->Next())
				object->DataSet(object->Next());
			LoadRecord(Last(), ++topRecordShowing + numberShowing - 1);
		}
		break;

	case L_PGUP:
		if (First() != Current())
			Add(First());
		else
		{
			topRecordShowing -= numberShowing;
			if (topRecordShowing < 0)
				topRecordShowing = 0;
			int recordNumber = topRecordShowing;
			for (VIRTUAL_ELEMENT *object = First(); object; object = object->Next())
				LoadRecord(object, recordNumber++);
		}
		break;

	case L_PGDN:
		if (Last() != Current())
			Add(Last());
		else
		{
			topRecordShowing += numberShowing;
			if (topRecordShowing > numberOfRecords - 1)
				topRecordShowing = numberOfRecords - 1;
			int recordNumber = topRecordShowing;
			for (VIRTUAL_ELEMENT *object = First(); object; object = object->Next())
				LoadRecord(object, recordNumber++);
		}
		break;

	case L_TOP:
		{
		topRecordShowing = 0;
		int recordNumber = topRecordShowing;
		for (VIRTUAL_ELEMENT *object = First(); object; object = object->Next())
			LoadRecord(object, recordNumber++);
		Add(First());
		}
		break;

	case L_BOTTOM:
		{
		topRecordShowing = numberOfRecords - 1;
		int recordNumber = topRecordShowing;
		for (VIRTUAL_ELEMENT *object = First(); object; object = object->Next())
			LoadRecord(object, recordNumber++);
		Add(Last());
		}
		break;

	default:
		ccode = UIW_WINDOW::Event(event);
	}

	// Return the control code.
	return (ccode);
}

void VIRTUAL_LIST::LoadRecord(VIRTUAL_ELEMENT *element, int recordNumber)
{
	// Load the record from the file.
	if (recordNumber > numberOfRecords - 1)
		element->DataSet(-1, "");
	else
	{
		long offset = recordLength * recordNumber;
		fseek(file, offset, SEEK_SET);
		char *text = element->DataGet();
		fgets(text, recordLength - 1, file);
		text[recordLength - 1] = '\0';
		element->DataSet(recordNumber, text);
	}
}

int UI_APPLICATION::Main(void)
{
	// The UI_APPLICATION constructor automatically initializes the 
	// display, eventManager, and windowManager variables.

	// Create a window containing the virtual list and attach it to the window manager.
	*windowManager
		+ &(*UIW_WINDOW::Generic(5, 2, 75, 12, "Dictionary")
			+ new VIRTUAL_LIST("vlist.txt", 78+EOL_LENGTH));

	// Wait for user response.
	UI_EVENT event;
	EVENT_TYPE ccode;
	do
	{
		// Get input from the user.
		eventManager->Get(event);

		// Send event information to the window manager.
		ccode = windowManager->Event(event);
	} while (ccode != L_EXIT && ccode != S_NO_OBJECT);

	return (0);
}
