//	FILEEDIT.CPP (FILEEDIT) - File editor example program.
//	COPYRIGHT (C) 1990-1993.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/

//  May be freely copied, used and distributed.

#include <fcntl.h>
#include <string.h>
#include <ui_win.hpp>
#include "file.hpp"
#include "direct.hpp"

#define USE_HELP_CONTEXTS
#include "filehelp.hpp"

// Maximum file size that can be read in.
const int MAX_FILE_LEN = 20000;

class EDIT_WINDOW : public UIW_WINDOW
{
public:
	EDIT_WINDOW(int left, int top, int width, int height);
	~EDIT_WINDOW(void);

	EVENT_TYPE Event(const UI_EVENT &event);

private:
	static EVENT_TYPE New(UI_WINDOW_OBJECT *object, UI_EVENT &event,
		EVENT_TYPE ccode);
	static EVENT_TYPE Open(UI_WINDOW_OBJECT *object, UI_EVENT &event,
		EVENT_TYPE ccode);
	static EVENT_TYPE Save(UI_WINDOW_OBJECT *object, UI_EVENT &event,
		EVENT_TYPE ccode);
	static EVENT_TYPE SaveAs(UI_WINDOW_OBJECT *object, UI_EVENT &event,
		EVENT_TYPE ccode);
	static EVENT_TYPE Stats(UI_WINDOW_OBJECT *object, UI_EVENT &event,
		EVENT_TYPE ccode);
	static EVENT_TYPE Help(UI_WINDOW_OBJECT *object, UI_EVENT &event,
		EVENT_TYPE ccode);

	DOS_FILE *file;
	UIW_TEXT *textField;
	char *editBuffer;
	UIW_TITLE *title;
};

EDIT_WINDOW::EDIT_WINDOW(int left, int top, int width, int height) :
	UIW_WINDOW(left, top, width, height, WOF_NO_FLAGS, WOAF_NO_FLAGS),
	file(0)
{
	// Create the buffer and text field to display and edit the buffer.
	editBuffer = new char[MAX_FILE_LEN + 1];
	editBuffer[0] = '\0';
	textField = new UIW_TEXT(0, 0, 0, 0, editBuffer, MAX_FILE_LEN, WNF_NO_WRAP,
		WOF_NON_FIELD_REGION | WOF_NO_ALLOCATE_DATA);
	*textField
		+ new UIW_SCROLL_BAR(0,0,0,0, SBF_CORNER)
		+ new UIW_SCROLL_BAR(0,0,0,0, SBF_VERTICAL)
		+ new UIW_SCROLL_BAR(0,0,0,0, SBF_HORIZONTAL);

	// Create the title to hold the file name.
	title = new UIW_TITLE("(untitled)", WOF_JUSTIFY_CENTER);

	// Add all window objects to the window.
	*this
		+ new UIW_BORDER
		+ new UIW_MAXIMIZE_BUTTON
		+ new UIW_MINIMIZE_BUTTON
		+ UIW_SYSTEM_BUTTON::Generic()
		+ title
		+ &(*new UIW_PULL_DOWN_MENU()
			+ &(*new UIW_PULL_DOWN_ITEM("&File")
				+ new UIW_POP_UP_ITEM("&New", MNIF_NO_FLAGS, BTF_NO_3D, WOF_NO_FLAGS, EDIT_WINDOW::New)
				+ new UIW_POP_UP_ITEM("&Open...", MNIF_NO_FLAGS, BTF_NO_3D, WOF_NO_FLAGS, EDIT_WINDOW::Open)
				+ new UIW_POP_UP_ITEM("&Save", MNIF_NO_FLAGS, BTF_NO_3D, WOF_NO_FLAGS, EDIT_WINDOW::Save)
				+ new UIW_POP_UP_ITEM("Save &As...", MNIF_NO_FLAGS, BTF_NO_3D, WOF_NO_FLAGS, EDIT_WINDOW::SaveAs)
				+ new UIW_POP_UP_ITEM("&File Stats...", MNIF_NO_FLAGS, BTF_NO_3D, WOF_NO_FLAGS, EDIT_WINDOW::Stats)
				+ new UIW_POP_UP_ITEM
				+ new UIW_POP_UP_ITEM("E&xit", MNIF_SEND_MESSAGE, BTF_NO_3D, WOF_NO_FLAGS, NULL, L_EXIT))
			+ &(*new UIW_PULL_DOWN_ITEM("&Help")
				+ new UIW_POP_UP_ITEM("&General help", MNIF_NO_FLAGS, BTF_NO_3D, WOF_NO_FLAGS, EDIT_WINDOW::Help)
				+ new UIW_POP_UP_ITEM
				+ new UIW_POP_UP_ITEM("&About", MNIF_NO_FLAGS, BTF_NO_3D, WOF_NO_FLAGS, EDIT_WINDOW::Help)))
		+ textField;
}

EDIT_WINDOW::~EDIT_WINDOW(void)
{
	// Delete the edit buffer used for holding the file text.
	if (editBuffer)
		delete editBuffer;

	// Delete (close) the file if open.
	if (file)
		delete file;
}

EVENT_TYPE EDIT_WINDOW::Event(const UI_EVENT &event)
{
	// Process the file selected message sent from the file selector.
	switch (event.type)
	{
	case FILE_NEW:
		{
		if (file)
			delete file;
		file = 0;
		title->DataSet("(untitled)");
		editBuffer[0] = '\0';
		textField->DataSet(editBuffer);
		break;
		}
	case FILE_FOPEN:
		{
		DOS_FILE *tFile = new DOS_FILE((char *)event.data);

		int noOfBytes = tFile->Read(editBuffer, MAX_FILE_LEN);
		if (noOfBytes == -1)
		{
			delete tFile;
			tFile = 0;
		}
		else
			editBuffer[noOfBytes] = '\0';
		// Put the file name in the title and update text field.
		if (tFile)
		{
			if (file)
			{
				file->Close();
				delete file;
			}
			file = tFile;
			title->DataSet((char *)event.data);
			textField->DataSet(editBuffer);
		}
		break;
		}
	case FILE_SAVE_AS:
		{
		DOS_FILE *tFile = new DOS_FILE((char *)event.data);
		if (tFile)
		{
			if (tFile->Open(O_CREAT) == -1)
				break;
			tFile->Write(editBuffer, strlen(editBuffer));

			// Put the file name in the title and update text field.
			if (file)
			{
				file->Close();
				delete file;
			}
			file = tFile;
			title->DataSet((char *)event.data);
		}
		break;
		}
	case FILE_DELETE:
		{
		break;
		}
	default:
		return (UIW_WINDOW::Event(event));
	};

	return event.type;
}

EVENT_TYPE EDIT_WINDOW::Help(UI_WINDOW_OBJECT *object, UI_EVENT &, EVENT_TYPE ccode)
{
	if (ccode != L_SELECT)
		return ccode;

	// Find the particular help context.
	int helpContext;
	const char *string = ((UIW_POP_UP_ITEM *)object)->DataGet();
	switch(string[2])
	{
	case 'G':
		helpContext = HELP_GENERAL;
		break;

	case 'A':
		helpContext = HELP_ABOUT;
		break;
	}

	// Call the help system to display help.
	helpSystem->DisplayHelp(object->windowManager, helpContext);
	return ccode;
}

EVENT_TYPE EDIT_WINDOW::New(UI_WINDOW_OBJECT *object, UI_EVENT &, EVENT_TYPE ccode)
{
	if (ccode != L_SELECT)
		return ccode;

	// Get a pointer to the current edit window.
	for(UI_WINDOW_OBJECT *topWindow = object; topWindow->parent; )
		topWindow = topWindow->parent;

	EDIT_WINDOW *editWindow = (EDIT_WINDOW *)topWindow;
	UI_EVENT tEvent;
	tEvent.type = FILE_NEW;
	editWindow->Event(tEvent);
	return ccode;
}

EVENT_TYPE EDIT_WINDOW::Open(UI_WINDOW_OBJECT *object, UI_EVENT &, EVENT_TYPE ccode)
{
	if (ccode != L_SELECT)
		return ccode;

	// Get a pointer to the current edit window.
	for(UI_WINDOW_OBJECT *topWindow = object; topWindow->parent; )
		topWindow = topWindow->parent;

	EDIT_WINDOW *editWindow = (EDIT_WINDOW *)topWindow;
	*object->windowManager + new FILE_WINDOW(FILE_FOPEN, (UIW_WINDOW *)editWindow);
	return ccode;
}

EVENT_TYPE EDIT_WINDOW::Stats(UI_WINDOW_OBJECT *object, UI_EVENT &, EVENT_TYPE ccode)
{
	if (ccode != L_SELECT)
		return ccode;

	// Get a pointer to the current edit window.
	for(UI_WINDOW_OBJECT *topWindow = object; topWindow->parent; )
		topWindow = topWindow->parent;

	EDIT_WINDOW *editWindow = (EDIT_WINDOW *)topWindow;
	*object->windowManager + new FILE_WINDOW(FILE_STATS, (UIW_WINDOW *)editWindow);
	return ccode;
}

EVENT_TYPE EDIT_WINDOW::Save(UI_WINDOW_OBJECT *object, UI_EVENT &event, EVENT_TYPE ccode)
{
	if (ccode != L_SELECT)
		return ccode;

	// Get a pointer to the current edit window.
	for(UI_WINDOW_OBJECT *topWindow = object; topWindow->parent; )
		topWindow = topWindow->parent;

	EDIT_WINDOW *editWindow = (EDIT_WINDOW *)topWindow;

	// If a file exists then write, otherwise get a file name.
	if (editWindow->file)
	{
		editWindow->file->Open(O_CREAT | O_TRUNC);
		editWindow->file->Write(editWindow->editBuffer, strlen(editWindow->editBuffer));
	}
	else
		SaveAs(object, event, ccode);
	return ccode;
}

EVENT_TYPE EDIT_WINDOW::SaveAs(UI_WINDOW_OBJECT *object, UI_EVENT &, EVENT_TYPE ccode)
{
	if (ccode != L_SELECT)
		return ccode;

	// Get a pointer to the current edit window.
	for(UI_WINDOW_OBJECT *topWindow = object; topWindow->parent; )
		topWindow = topWindow->parent;

	EDIT_WINDOW *editWindow = (EDIT_WINDOW *)topWindow;
	*object->windowManager + new FILE_WINDOW(FILE_SAVE_AS, (UIW_WINDOW *)editWindow);
	return ccode;
}

int UI_APPLICATION::Main(void)
{
	// The UI_APPLICATION constructor automatically initializes the 
	// display, eventManager, and windowManager variables.

	// Initialize the default storage.
	UI_STORAGE *storage = new UI_STORAGE("dir_win.dat", UIS_READ);
	UI_WINDOW_OBJECT::defaultStorage = storage;

	// Add the edit window to the center of the screen.
	*windowManager + new EDIT_WINDOW(3, 1, 60, 15);

	//Initialize the error and help window system.
	UI_WINDOW_OBJECT::helpSystem = new UI_HELP_SYSTEM("filehelp.dat", windowManager, HELP_GENERAL);

	// Wait for user response.
	UI_EVENT event;
	EVENT_TYPE ccode;
	do
	{
		// Get input from the user.
		eventManager->Get(event);

		// Send event information to the window manager.
		ccode = windowManager->Event(event);
	} while (ccode != L_EXIT && ccode != S_NO_OBJECT);

	// Clean up.
 	delete UI_WINDOW_OBJECT::helpSystem;

	return (0);
}
